
/*
 Emulator DCF77
 Simulate a DCF77 radio receiver with a ESP8266, esp01 model
 Emit a complete three minute pulses train from the GPIO2 output
 the train is preceded by a single pulse ad the lacking 59 pulse to allow some clock 
 model sincronization  of the beginning frame
 after the three pulses train one more single pulse is sent to safely close the frame
 get the time from the ntp service

 Uses Time library to facilitate time computation
 
 Known issue:
 -When the DayLightSaving mode change, the three minutes packet is not checked for 
  possible changes of hour across the frame itself, moreover the daylight saving is 
  changed normally at 3 o clock in the morning (here in italy), while I don't correct 
  for it so the time will be probably incorrect before at least 03:03 then dayLightSaving 
  changes.
 -The exact "second" precision is not guaranteed because of the semplicity of the NTP 
  imnplementation normally the packet transit delay would be taken into account, but here 
  is not.

 Fuso68 05/12/2015

 19/12/2015 added disconnect and reconnect if wlan fail, reconnect also after three 
 .......... failed ntp requests
 20/12/2015 increased wlan connect timeout to 30 seconds
 25/03/2018 correct TimeDayOfWeekeek to match TimeLibrary format with DCF77 format, 
 .......... see TimeDayOfWeekeek section in CalculateArray
   
 Based upon:
 Udp NTP Client

 Get the time from a Network Time Protocol (NTP) time server
 Demonstrates use of UDP sendPacket and ReceivePacket
 For mTimeHours on NTP time servers and the messages needed to communicate with them,
 see http://en.wikipedia.org/wiki/Network_Time_Protocol

 created 4 Sep 2010
 by Michael Margolis

 modified 9 Apr 2012
 by Tom Igoe

 updated for the ESP8266 12 Apr 2015 
 by Ivan Grokhotkov

 Updated 27.Oct 2022 with WiFiManager and Deep-Sleep (1 h) and 
 clearing wifi-credencials by pulling GPIO4 (D2) to GND at start 
 by Michael Schober

 Amatronik (supported by ChatGPT)
 modified November 20, 2024
 Adapted for use with NodeMCU (ESP8266, circuit board DSW)
 - Option DCF signal phase
 - Option reduced waiting time (Deep-Sleep)
 - time control output (open collector output)
 - LED for time control output
 by Ingolf Bauer (amatronik@arcor.de)

 This code is in the public domain.
*/
//########################################################################
//#define DEBUG_MODE         // Uncomment this line to enable debug output
//########################################################################

#include <ESP8266WiFi.h>
#include <Ticker.h>
#include <TimeLib.h>
#include <WiFiManager.h> 
#include <WiFiUdp.h>

/* operating without the “WifiManager”
 char ssid[] = "SSID";                 // your network SSID (name)
 char pass[] = "VerySecret";           // your network password
*/
WiFiManager wifiManager;               // to store Wifi-Credencials
unsigned int localPort = 2390;         // local port to listen for UDP packets

/* 
 Don't hardwire the IP address or we won't get the benefits of the pool.
 Lookup the IP address for the host name instead 
*/

IPAddress timeServerIP;                // NTP server address
/*
 examples of possible entries:
 const char* ntpServerName = "fritz.box";
 const char* ntpServerName = "time.nist.gov";
*/
const char* ntpServerName = "ptbtime3.ptb.de";

const int NTP_PACKET_SIZE = 48;        // NTP time stamp is in the first 48 bytes of the message
byte packetBuffer[ NTP_PACKET_SIZE];   // buffer to hold incoming and outgoing packets
WiFiUDP udp;                           // a UDP instance to let us send and receive packets over UDP
int UdpNoReplyCounter = 0;             // udp reply missing counter
Ticker DcfOutTimer;                    // routine timer ogni 100 msec

/*
 LedPin 2 (D4) blinks at every dcf77-bin incoming.
 It is on until a dcf77-Minute starts. Afterwards it blinks continous by every dfc77-bit incoming.
 It blinks continuous 3-times a second if clearing of wifi-credencials be requested..
 .. by Pin 4 (D2) ist pulled to ground.

 If Pin 4 (D2) is Ground, the WiFi-Credencials will be deleted in the eeprom,
 when the "WifiManager" is used
*/
#define LedPin 2             // GPIO has a pullup-Resistor
#define ClearEepromPin 4              
/*
Amatronik
 If Pin 14 (D5) is Ground, the DCF output signal is inverted
 If Pin 12 (D6) is Ground, the cycle time is reduced
 .. Pin  5 (D1) is H/L .., Signal output for the actuator at Pin 13 (D7)  
 .. Pin 13 (D7) is H/L .., Signal output for the timer relay (open collector)
*/
#define Ssiv 14              // Switch SIV, GPIO has a pullup-Resistor
#define Scyt 12              // Switch CYT, GPIO has a pullup-Resistor
#define ZAZ   5              // Signaling for the actor
#define ZSR  13              // Actuator (e.g. relay) can be connected

/*
Amatronik
 With this version two time windows can be implemented.
 Definition of two times at which the timer relay should be switched on or off.
*/
String timeOn1  = "10:02";   // 1. Switch-on time
String timeOff1 = "10:03";   // 1. Switch-off time
String timeOn2  = "20:00";   // 2. Switch-on time
String timeOff2 = "20:15";   // 2. Switch-off time

bool flagOn = false;
bool flagOff = false;

/*
If the monitor does not output all messages, there are several ways to 
change this. 
One method is to create a wrapper class that extends the standard Serial 
methods and Serial.flush(); integrated.
Use in code: 
- MySerial.print("...");
- MySerial.println("...");
Or you can use an external serial monitor like CoolTerm, Tera Term or PuTTY to
display the data faster as they may have lower latency in displaying the data.
*/

class SerialWrapper {
public:
    SerialWrapper() : previousMillis(0), interval(1000) {} // set the interval to 1 second
    
    void println(const String& message) {
      Serial.println(message);
      flushIfNeeded();
    }
    
    void print(const String& message) {
      Serial.print(message);
      flushIfNeeded();
    }

private:
    unsigned long previousMillis;      // Time of the last flush call
    const long interval;               // Time interval for the flush call (in milliseconds)

// This method ensures that flush() is only called at regular intervals.
    void flushIfNeeded() {
      unsigned long currentMillis = millis();
      
// Check whether there is a connection and whether enough time has passed.
        if (Serial && currentMillis - previousMillis >= interval) {
          Serial.flush();                        // If connected, execute Serial.flush()
          previousMillis = currentMillis;        // update the time
        }
    }
};

// Instance of the wrapper class
SerialWrapper MySerial;

/*
 how many total pulses we have
 three complete minutes + 2 head pulses and one tail pulse
*/
#define MaxPulseNumber 183
#define FirstMinutePulseBegin 2
#define SecondMinutePulseBegin 62
#define ThirdMinutePulseBegin 122

/*
 complete array of pulses for three minutes
 0 = no pulse, 1 = 100 msec, 2 = 200 msec 
*/
int ArrayImpulses[MaxPulseNumber];

int PulseCounter = 0;
int OutputDcfOn = 0;
int PulseCounterPartial = 0;
int TimeHours,TimeMinutes,TimeSeconds,TimeDay,TimeMonth,TimeYear,TimeDayOfWeek;

const int timeZone = 1;                // Central European Time

int Dls;                               // DayLightSaving
int tick;                              // For blinking every second while waiting until next ntp-time

/*
 Amatronik - Cycle times in minutes
 Please note that if you change the waiting time until the time log is transmitted just one night, 
 the currently valid waiting time must have expired. an immediate change can only be achieved with
 a restart.
*/
bool flagStatus = false;               // status waiting time flag 
int CycleTicks;                        // waiting time
int switchState;                       // Variable to store the switch status
int defaultCycleTimeMinutes = 60;      // Default waiting time:   1 hour    (double flashing of the LED)  
int reducedCycleTimeMinutes = 1;       // Shortened waiting time: 1 minutes (simple flashing of the LED)

void setup()
{
  Serial.begin(115200);
  Serial.println();

  #ifdef DEBUG_MODE
   Serial.println("DCF77 signal generator 2024-11-27");
  #endif

// when forced to GND = LOW, otherwise = HIGH
  pinMode(ClearEepromPin, INPUT_PULLUP);
  
  pinMode(LedPin, OUTPUT);
  digitalWrite(LedPin, LOW);

/*
 Amatronik
 Determination of the input and output channels
*/
  pinMode(Scyt, INPUT);                // sets the GPIO pin as input
  pinMode(Scyt, INPUT_PULLUP);         // activate pullup resistor
  pinMode(Ssiv, INPUT);                // sets the GPIO pin as input
  pinMode(Ssiv, INPUT_PULLUP);         // activate pullup resistor 
  
  pinMode(ZAZ, OUTPUT);
  digitalWrite(ZAZ, HIGH);             // Switch off the channel first
  pinMode(ZSR, OUTPUT);
  digitalWrite(ZSR, LOW);              // Switch off the channel first

/*
  If GPIO4 (D2) = Ground at start then erease Wifi-Credencials and blink 3-times/s until switch off.
  Newstart needed!
*/
  if (digitalRead(ClearEepromPin) == LOW){
    wifiManager.resetSettings();
    while(1==1){
      digitalWrite(LedPin, HIGH);
      delay(300);
      digitalWrite(LedPin, LOW);
      delay(300);  
    }
  }
// wifiManager.resetSettings();                            // only uncomment if you want to reset store Credencials
  wifiManager.autoConnect("DCF77-Signal Generator");       // Creates an access-point on 192.168.4.1
  
  DcfOutTimer.attach_ms(100, DcfOut);            // DCF pulse management

/*
 first 2 pulses: 1 + blank to simulate the packet beginning
 the first bit is a 1
*/
  ArrayImpulses[0] = 1;
// follows the missing pulse which indicates the start-of-minute search synchronism
  ArrayImpulses[1] = 0;
  ArrayImpulses[MaxPulseNumber - 1] = 1;         // last pulse after the third 59th blank
  
  PulseCounter = 0;
  OutputDcfOn = 0;                               // we begin with the output OFF

// ConnectWifi();
}

void loop(){
  if (WiFi.status() == WL_CONNECTED)             // check the lan status
     LeggiEdecodificaTempo();
  else
// ConnectWifi();
// wait 1 Minute
// delay(60000);
// wait 1 hour
    #ifdef DEBUG_MODE
     Serial.println("waiting time (e.g. one hour), LED will blink repeately every 4s.");
    #endif
/* 
Amatronik
Checking and processing the current time for the timer relay
*/
   switchZSR();

// Pause time configuration
  bool switchState = digitalRead(Scyt);          // Query the switch position

  if (switchState == LOW) {       
   CycleTicks = (reducedCycleTimeMinutes * 60) / 4;
   flagStatus = true;                            // low level at the input
   } else {
     CycleTicks = (defaultCycleTimeMinutes * 60) / 4;
     flagStatus = false;                         // high level at the input
  } 

  for (tick = 0; tick < CycleTicks; tick++){
    if (tick == 0){          // Amatronik     
      delay(45000);          // wait until the last telegram has been completely transmitted
    }

// Amatronik - different statuses of the LED to differentiate    
    digitalWrite(LedPin, HIGH);                  // Switch LED off
    if (flagStatus = false) {                    // reduced waiting time
     delay(3900);
     digitalWrite(LedPin, LOW);                  // Switch LED on
     delay(100);
    } else {                                     // default waiting time
       delay(3700); 
       digitalWrite(LedPin, LOW);                // Switch LED on
       delay(100);
       digitalWrite(LedPin, HIGH);               // Switch LED off
       delay(100);
       digitalWrite(LedPin, LOW);                // Switch LED on
       delay(100);
    }    
// Amatronik - check and process the current time
  switchZSR();                                   // query time window for output channel
  }
  yield();
  digitalWrite(LedPin, HIGH);                    // Switch LED off
}

void LeggiEdecodificaTempo() {
  int DayToEndOfMonth,TimeDayOfWeekeekToEnd,TimeDayOfWeekeekToSunday;

  WiFi.hostByName(ntpServerName, timeServerIP);  // get a random server from the pool
  #ifdef DEBUG_MODE
   Serial.println("Starting UDP");
  #endif

  udp.begin(localPort);

  #ifdef DEBUG_MODE
   Serial.print("Local port: ");
   Serial.println(udp.localPort());
  #endif
  sendNTPpacket(timeServerIP);         // send an NTP packet to a time server
  delay(1000);                         // wait to see if a reply is available
  
  int cb = udp.parsePacket();
  if (!cb) {
  #ifdef DEBUG_MODE
    Serial.println("no packet yet");
  #endif
// try max 3 times (every minute) after that we force the wifi to reconnect
    if (UdpNoReplyCounter++ == 3){
      #ifdef DEBUG_MODE
       Serial.println("answer-mistake UDP");
      #endif

// ConnectWifi();
      UdpNoReplyCounter = 0;
    };
  } else {
    UdpNoReplyCounter = 0;

    #ifdef DEBUG_MODE
     Serial.print("packet received, length=");
     Serial.println(cb);
    #endif

// We've received a packet, read the data from it
    udp.read(packetBuffer, NTP_PACKET_SIZE);     // read the packet into the buffer

/*
  the timestamp starts at byte 40 of the received packet and is four bytes,
  or two words, long. First, esxtract the two words:
*/
    unsigned long highWord = word(packetBuffer[40], packetBuffer[41]);
    unsigned long lowWord = word(packetBuffer[42], packetBuffer[43]);
    
/* 
 combine the four bytes (two words) into a long integer
 this is NTP time (seconds since Jan 1 1900)
*/
    unsigned long secsSince1900 = highWord << 16 | lowWord;
    
    #ifdef DEBUG_MODE
     Serial.print("Seconds since Jan 1 1900 = " );
     Serial.println(secsSince1900);
    #endif

// now convert NTP time into everyday time:
    #ifdef DEBUG_MODE
     Serial.print("Unix time = ");
    #endif
     
// Unix time starts on Jan 1 1970. In seconds, that's 2208988800:
    const unsigned long seventyYears = 2208988800UL;

/* 
 subtract seventy years
 note: 
 we add two minutes because the dcf protocol send the time of the FOLLOWING minute
 and our transmissione begin the next minute mTimeHours
*/
    time_t epoch = secsSince1900 - seventyYears + ( timeZone * 3600 ) + 120;

// Set the internal time on the ESP8266 using the calculated epoch time
    setTime(epoch);

// print Unix time:
    #ifdef DEBUG_MODE
     Serial.println(epoch);
    #endif 

// calculate actual day to evaluate the summer/winter time of day ligh saving
    TimeDayOfWeek = weekday(epoch);
    TimeDay = day(epoch);
    TimeMonth = month(epoch);
    TimeYear = year(epoch);

    #ifdef DEBUG_MODE
     Serial.print("Local Time ");      // UTC is the time at Greenwich Meridian (GMT)
     Serial.print(TimeDay);
     Serial.print('/');
     Serial.print(TimeMonth);
     Serial.print('/');
     Serial.print(TimeYear);
     Serial.print("\nDayOfWeekeek 1=Sunday 7=Saturday :");
     Serial.print(TimeDayOfWeek);
     Serial.print(' ');
    #endif
      
// solar or summer time calculation
    Dls = 0;       // default winter time

// from April to september we are surely on summer time
    if (TimeMonth > 3 && TimeMonth < 10) {
      Dls = 1;
    };

/*
 March, month of change winter->summer time, last last sunday of the month
 March has 31 days so from 25 included on sunday we can be in summer time
*/
    if (TimeMonth == 3 && TimeDay > 24) {
      DayToEndOfMonth = 31 - TimeDay;
      TimeDayOfWeekeekToSunday = 7 - TimeDayOfWeek;
      if (TimeDayOfWeekeekToSunday >= DayToEndOfMonth)
        Dls = 1;
    };

/* 
 Octobee, month of change summer->winter time, the last Sunday of the month
 Even Octobee has 31 days so from 25 included on sunday we can be in winter time
*/
    if (TimeMonth == 10) {
      Dls = 1;
      if (TimeDay > 24) {
        DayToEndOfMonth = 31 - TimeDay;
        TimeDayOfWeekeekToEnd = 7 - TimeDayOfWeek;
        if (TimeDayOfWeekeekToEnd >= DayToEndOfMonth)
        Dls = 0;
      };
    };

    #ifdef DEBUG_MODE
     Serial.print(" Dls:");
     Serial.print(Dls);
     Serial.print(' ');
    #endif

// add one hour if we are in summer time
    if (Dls == 1)
      epoch += 3600;

/*
 now that we know the dls state, we can calculate the time too
 print the hour, minute and second
*/
      TimeHours = hour(epoch);
      TimeMinutes = minute(epoch);
      TimeSeconds = second(epoch);

      #ifdef DEBUG_MODE
       Serial.print(TimeHours);        // print the hour
       Serial.print(':');
       Serial.print(TimeMinutes);      // print the minute
       Serial.print(':');
       Serial.println(TimeSeconds);    // print the second
      #endif 
   
/*
 If we are over about the 56th second we risk to begin the pulses too late, so it's better
 to skit at the half of the next minute and NTP+recalculate all again.
*/
    if (TimeSeconds > 56){
      delay(30000);
      return;      
    }

    CalculateArray(FirstMinutePulseBegin);       // calculate bits array for the first minute
    
    epoch += 60;             // add one minute ad calculate array again fot the second minute
    TimeDayOfWeek = weekday(epoch);
    TimeDay = day(epoch);
    TimeMonth = month(epoch);
    TimeYear = year(epoch);
    TimeHours = hour(epoch);
    TimeMinutes = minute(epoch);
    TimeSeconds = second(epoch);
    CalculateArray(SecondMinutePulseBegin);

    epoch += 60;             // one minute mTimeHours for the third minute
    TimeDayOfWeek = weekday(epoch);
    TimeDay = day(epoch);
    TimeMonth = month(epoch);
    TimeYear = year(epoch);
    TimeHours = hour(epoch);
    TimeMinutes = minute(epoch);
    TimeSeconds = second(epoch);
    CalculateArray(ThirdMinutePulseBegin);
   
/*
 how many to the minute end ?
 don't forget that we begin transmission at second 58
*/
    int DaPerdere = 58 - TimeSeconds;
    delay(DaPerdere * 1000);
  
    OutputDcfOn = 1;         //begin

/*
 three minutes are needed to transmit all the packet,
 then wait mTimeHours 30 secs to locate safely at the half of minute
 NB 150+ 60= 210 sec, 60 secs are lost from main routine
*/
    delay(150000);
  };
  udp.stop() ;
}

void CalculateArray(int ArrayOffset) {
  int n,Tmp,TmpIn;
  int ParityCount = 0;

// we set the first 20 bits of each minute to logical zero
  for (n=0;n<20;n++)
    ArrayImpulses[n+ArrayOffset] = 1;

  if (Dls == 1)                        // DayLightSaving bit
    ArrayImpulses[17+ArrayOffset] = 2;
  else
    ArrayImpulses[18+ArrayOffset] = 2;
    
 
  ArrayImpulses[20+ArrayOffset] = 2;   // bit 20 must be 1 to indicate active time

  TmpIn = Bin2Bcd(TimeMinutes);        // calculate the bits per minute
  for (n=21;n<28;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  };
  if ((ParityCount & 1) == 0)
    ArrayImpulses[28+ArrayOffset] = 1;
  else
    ArrayImpulses[28+ArrayOffset] = 2;

    ParityCount = 0;                   // calculates bits for hours
    TmpIn = Bin2Bcd(TimeHours);

  for (n=29;n<35;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  }
  if ((ParityCount & 1) == 0)
    ArrayImpulses[35+ArrayOffset] = 1;
  else
    ArrayImpulses[35+ArrayOffset] = 2;
    ParityCount = 0;
    TmpIn = Bin2Bcd(TimeDay);          //calculate the bits for the day

  for (n=36;n<42;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  }
/*
 calculate the bits for the day of the week
 the next line correct format to match DCF77, 
 in TimeLibrary Sunday = 1 Saturday = 7, in DCF77 Monday = 1 Sunday = 7
*/
  TimeDayOfWeek -= 1;  
  if (TimeDayOfWeek == 0) TimeDayOfWeek = 7;
  TmpIn = Bin2Bcd(TimeDayOfWeek);
  for (n=42;n<45;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  }

  TmpIn = Bin2Bcd(TimeMonth);          // calculate the bits for the month
  for (n=45;n<50;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  }

// calculate the bits for the year
  TmpIn = Bin2Bcd(TimeYear - 2000);    // we are only interested in the year with... the millenniumbug!
  for (n=50;n<58;n++) {
    Tmp = TmpIn & 1;
    ArrayImpulses[n+ArrayOffset] = Tmp + 1;
    ParityCount += Tmp;
    TmpIn >>= 1;
  }
  
  if ((ParityCount & 1) == 0)          // date parity
    ArrayImpulses[58+ArrayOffset] = 1;
  else
    ArrayImpulses[58+ArrayOffset] = 2;

  ArrayImpulses[59+ArrayOffset] = 0;   // last missing impulse

/*
 for debug: print the whole 180 secs array
 Serial.print(':');
 for (n=0;n<60;n++)
 Serial.print(ArrayImpulses[n+ArrayOffset]);
*/
}

// send an NTP request to the time server at the given address
unsigned long sendNTPpacket(IPAddress& address)
{
  #ifdef DEBUG_MODE
   Serial.println("sending NTP packet...");
  #endif 

  memset(packetBuffer, 0, NTP_PACKET_SIZE);      // set all bytes in the buffer to 0
/*
 Initialize values needed to form NTP request
 (see URL above for details on the packets)
*/
  packetBuffer[0] = 0b11100011;                  // LI, Version, Mode
  packetBuffer[1] = 0;                           // Stratum, or type of clock
  packetBuffer[2] = 6;                           // Polling Interval
  packetBuffer[3] = 0xEC;                        // Peer Clock Precision

// 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12]  = 49;
  packetBuffer[13]  = 0x4E;
  packetBuffer[14]  = 49;
  packetBuffer[15]  = 52;

/*
 all NTP fields have been given values, now
 you can send a packet requesting a timestamp
*/
  udp.beginPacket(address, 123);                 // NTP requests are to port 123
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();
  return(100);
}

/*
 recalled cyclically every 100 msec
 for DCF77 simulation output
*/
void DcfOut() {

  if (OutputDcfOn == 1) {
    bool invertSignal = digitalRead(Ssiv);                 // read the level of the GPIO pin

    switch (PulseCounterPartial++) {
      case 0:
        if (ArrayImpulses[PulseCounter] != 0)
// digitalWrite(LedPin, 1);
          digitalWrite(LedPin, invertSignal ? 0 : 1);      // negate if signal is true 
        break;
      case 1:
        if (ArrayImpulses[PulseCounter] == 1)
// digitalWrite(LedPin, 0);
          digitalWrite(LedPin, invertSignal ? 1 : 0);      // negate if signal is true
        break;
      case 2:
//digitalWrite(LedPin, 0);
          digitalWrite(LedPin, invertSignal ? 1 : 0);      // negate if signal is true
        break;
      case 9:
        if (PulseCounter++ == (MaxPulseNumber -1 )){       // one less because we FIRST tx the pulse THEN count it
          PulseCounter = 0;
          OutputDcfOn = 0;
        };
        PulseCounterPartial = 0;
        break;
    };
  };
}

int Bin2Bcd(int argument) {
  int msb,lsb;

  if (argument < 10)
  return argument;
  msb = (argument / 10) << 4;
  lsb = argument % 10; 
  return msb + lsb;
}

/*
 Amatronik
 Switching function based on current time
 time correction 2 minutes
*/
void switchZSR() {
  char currentTime[6];                 // get the current time as a string
  
  int currentHour = hour();            // get the current hour and minute
  int currentMinute = minute();

  currentMinute -= 2;                  // subtract 2 minutes (120 seconds)

  if (currentMinute < 0) {             // fix overflow (if minutes < 0)
    currentMinute += 60;
    currentHour -= 1;
  }
  if (currentHour < 0) {               // fix overflow for hours (if hours < 0)
    currentHour += 24;
  }
  sprintf(currentTime, "%02d:%02d", currentHour, currentMinute);

// check whether the current time is within the defined intervals
  if (isTimeInRange(timeOn1, timeOff1, currentTime) || isTimeInRange(timeOn2, timeOff2, currentTime)) {
    digitalWrite(ZSR, HIGH);           // turn on channel
    digitalWrite(ZAZ, LOW);            // turn on channel-LED

    #ifdef DEBUG_MODE
     if (String(currentTime) == timeOn1 || String(currentTime) == timeOn2) {
        if (!flagOn) {
          flagOn = true; 
          flagOff = false;
          MySerial.print(currentTime);
          MySerial.print(" : ");
          MySerial.println("ZSR turned on");
        }  
     }
    #endif

  } else {
    digitalWrite(ZSR, LOW);            // turn off channel
    digitalWrite(ZAZ, HIGH);           // turn off channel-LED

    #ifdef DEBUG_MODE
     if (String(currentTime) == timeOff1 || String(currentTime) == timeOff2) {  
        if (!flagOff) {
         flagOff = true;
         flagOn = false; 
         MySerial.print(currentTime);
         MySerial.print(" : ");
         MySerial.println("ZSR turned off");
        }   
      }
    #endif
  }
}

// function checks whether a time is within a range
bool isTimeInRange(String startTime, String endTime, String currentTime) {
  if (currentTime >= startTime && currentTime < endTime) {
    return true;
  }
  return false;
}

// end of program
